%CODEGENERATOR.GENDOTPRODC Generates a dot product C-implementation.
%
% CGen.gendotprodc generates a .h and a .c file in the directory
% specified by ccodepath.
%
% Notes::
% - Is called by geninvdyn if CGen has active flag genmex or genccode.
%
% Authors::
%  Joern Malzahn   (joern.malzahn@tu-dortmund.de) 
%
% See also CodeGenerator, CodeGenerator.gengaussjordanc.

% Copyright (C) 1993-2014, by Peter I. Corke
% Copyright (C) 2014, by Joern Malzahn
%
% This file is part of The Robotics Toolbox for Matlab (RTB).
%
% RTB is free software: you can redistribute it and/or modify
% it under the terms of the GNU Lesser General Public License as published by
% the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
%
% RTB is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU Lesser General Public License for more details.
%
% You should have received a copy of the GNU Lesser General Public License
% along with RTB.  If not, see <http://www.gnu.org/licenses/>.
%
% http://www.petercorke.com
function [] =  gendotprodc(CGen)

funname = 'dotprod';
funfilename = [funname,'.c'];
hfilename = [funname,'.h'];
srcDir = fullfile(CGen.ccodepath,'src');
hdrDir = fullfile(CGen.ccodepath,'include');

% Check for existance of dotProd C-files
if exist(fullfile(srcDir,funfilename),'file') && exist(fullfile(srcDir,funfilename),'file')
    return;
end

% Check for existance of C-code directories
if ~exist(srcDir,'dir')
    mkdir(srcDir);
end
if ~exist(hdrDir,'dir')
    mkdir(hdrDir);
end

fSignature = ['double ',funname,'(const double *input1, const double *input2, int nEl)'];
% Create the function description header
hStruct = createHeaderStruct(funname); % create header
hStruct.calls = fSignature;
if ~isempty(hStruct)
    hFString = CGen.constructheaderstringc(hStruct);
end

%% Generate C implementation file
fid = fopen(fullfile(srcDir,funfilename),'w+');

% Includes
fprintf(fid,'%s\n\n',...
    ['#include "', hfilename,'"']);
% Function
fprintf(fid,'%s\n',[fSignature,'{']);
fprintf(fid,'\t%s\n','double res = 0;');
fprintf(fid,'\t%s\n\n','int iEl = 0;');
fprintf(fid,'\t%s\n','for (iEl = 0; iEl < nEl; iEl++){');
fprintf(fid,'\t\t%s\n','res += input1[iEl] * input2[iEl];');
fprintf(fid,'\t%s\n\n','}');
fprintf(fid,'\t%s\n','return res;');
fprintf(fid,'%s\n','}');

fclose(fid);

%% Generate C header file
fid = fopen(fullfile(hdrDir,hfilename),'w+');

% Function description header
fprintf(fid,'%s\n\n',hFString);

% Include guard
fprintf(fid,'%s\n%s\n\n',...
    ['#ifndef ', upper([funname,'_h'])],...
    ['#define ', upper([funname,'_h'])]);

% Function prototype
fprintf(fid,'%s\n\n',['double ',funname,'(const double *input1, const double *input2, int nEl);']);

% Include guard
fprintf(fid,'%s\n',...
    ['#endif /*', upper([funname,'_h */'])]);

fclose(fid);

CGen.logmsg('\t%s\n',' done!');

end

 %% Definition of the function description header contents
function hStruct = createHeaderStruct(fname)
[~,hStruct.funName] = fileparts(fname);
hStruct.shortDescription = ['Compute the dot product of two vectors'];
hStruct.detailedDescription = {['Given two vectors of length nEl in the input arrays'],...
    'input1 and input 2 the function computes the dot product (or scalar product) of both vectors.'};
hStruct.inputs = { ['input1: nEl element array of doubles,'],...
                   ['input2: nEl element array of doubles,'],...
                    'nEl: dimension of the two arrays.'};
hStruct.outputs = {};
hStruct.references = {'The C Book: http://publications.gbdirect.co.uk/c_book/'};
hStruct.authors = {'This is an autogenerated function!',...
    'Code generator written by:',...
    'Joern Malzahn (joern.malzahn@tu-dortmund.de)'};
hStruct.seeAlso = {'CodeGenerator'};
end